#include <float.h>
#include <math.h>
#include <assert.h>

#include <maya/MPxFileTranslator.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnTransform.h>
#include <maya/MSelectionList.h>
#include <maya/MItSelectionList.h>
#include <maya/MBoundingBox.h>
#include <maya/MGlobal.h>
#include <maya/MTime.h>
#include <maya/MMatrix.h>
#include <maya/MItDependencyNodes.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MDagPathArray.h>
#include <maya/MFnIkHandle.h>
#include <maya/MObjectArray.h>
#include <maya/MFnBlendShapeDeformer.h>
#include <maya/MFnNurbsSurface.h>
#include <maya/MAnimControl.h>
#include <maya/MFloatArray.h>
#include <maya/MFnMesh.h>

#include <MDt.h>
#include <MDtExt.h>

#include "rwcommon.h"
#include "material.h"
#include "remapper.h"
#include "querymel.h"
#include "node.h"
#include "scene.h"
#include "blind.h"
#include "global.h"

/* exportScene member functions */

exportScene::exportScene()
{
    bool exportBlindData = false;

    numAnimatedNodes    = 0;
    lastChildIndex      = 0;
    nodeInvMatrices     = NULL;
    nodeAnimFlags       = NULL;
    nodeTags            = NULL;
    tagOffset           = 0;

    /* Check all the file nodes in the scene and see if any have K&L data attached */
    forAllNodesInScene(MFn::kFileTexture, fileTextureCallBack, (void *)&globalData->m_exportKL);

    /* Check if the scene has any blind data templates */
    processBlineDataTemplates();
    
    if (globalData->blindDataTemplates.size() > 0)
    {
        exportBlindData = true;
    }

    /* Initialise RenderWare plugins. */
    OpenRenderWare(NULL,
        globalData->m_exportRpAnim,
        (globalData->m_exportRpSkin || globalData->m_exportRpSkinWeighting),
        globalData->m_exportRpHAnim,
        false,
        globalData->m_morphTargets,
        false,
        globalData->m_exportKL,
        exportBlindData);

    initialiseMayaSdk();

    intialiseGlobalSceneData();

    nodes = new(sceneNodes);

    skinNodes = new(sceneNodes);
}

exportScene::~exportScene()
{
    nodeIt i;

    if (nodeInvMatrices)
    {
        RwFree(nodeInvMatrices);
        nodeInvMatrices = NULL;
    }

    if (nodeAnimFlags)
    {
        RwFree(nodeAnimFlags);
        nodeAnimFlags = NULL;
    }

    if (nodeTags)
    {
        RwFree(nodeTags);
        nodeTags = NULL;
    }

    /* Free up the list of scene nodes */
    for (i = nodes->begin(); i != nodes->end(); i++)
    {
        delete(*i);
    }

    delete(nodes);

    /* Also free up the seperate list of skinned scene nodes */
    for (i = skinNodes->begin(); i != skinNodes->end(); i++)
    {
        delete(*i);
    }

    delete (skinNodes);

    cleanupGlobalSceneData();

    cleanupMayaSdk();

    CloseRenderWare();
}

MStatus exportScene::exportDff(char *fileName)
{
    MStatus     status = MStatus::kSuccess;
    RpClump     *exportClump;
    char        animName[1024];
    char        *seperator;


    /* Find the node the user has selected */
    globalData->selectedMdtShape = findSelectedShape();

    /* Find the root node of the hierarchy the user has selected */
    rootMdtShape = findRootShape();

    /* Convert the selected hierarchy to a clump. */
    exportClump = convertHierarchyToClump();

    if (!exportClump)
    {
        status = MStatus::kFailure;
    }    

    if (globalData->m_exportRpSkin)
    {
        RpSkinAnim  *skinAnim = NULL;

        generateAnimFromHierarchyKeyFrames(rpSkinHierarchyKeys, &skinAnim, NULL);
        
        if (skinAnim != NULL)
        {
            strcpy(animName, fileName);
            seperator = strrchr(animName, '.');
            if (seperator)
            {
                strcpy(seperator, ".ska");
            }
            RpSkinAnimWrite(skinAnim, animName);
        }
    }
    
    if (globalData->m_exportRpHAnim)
    {
        RpHAnimAnimation    *hAnim = NULL;

        generateAnimFromHierarchyKeyFrames(rpHAnimHierarchyKeys, NULL, &hAnim);
        
        if (hAnim != NULL)
        {
            strcpy(animName, fileName);
            seperator = strrchr(animName, '.');
            if (seperator)
            {
                strcpy(seperator, ".anm");
            }
            RpHAnimAnimationWrite(hAnim, animName);
        }
    }

    if (globalData->m_verbose && exportClump)
    {
        printf("Conversion complete now saving\n");
    }

    if (exportClump)
    {
        if (!globalData->m_cullDff)
        {
            RwStream *stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMWRITE, fileName);

            RpClumpStreamWrite(exportClump, stream);
            RwStreamClose(stream, NULL);
        }
        
        RpClumpDestroy(exportClump);
    }

    /* return from the plugin command */
    return status;
}

RpClump * exportScene::convertHierarchyToClump()
{
    RpClump *clump = NULL;
    RwFrame *frame;

    if (rootMdtShape != -1)
    {
        RwFrame *clumpFrame;
        int     atomicCount, i, selectionRootIndex;

        
        globalData->hierarchyRootShape = rootMdtShape;
        clump = RpClumpCreate();
        frame = RwFrameCreate();

        /*
            Traverse the selected hierarchy and build up an array of sceneNodes that
            represent it.
        */
        nodeCounter = 0;

        preProcessHierarchy(rootMdtShape);

        /*
            Run through the scene nodes and attempt to merge geometries and remove
            unnecessary nodes.
        */
        optimiseHierarchy();

        if (globalData->m_exportRpAnim ||
            globalData->m_exportRpSkin ||
            globalData->m_exportRpHAnim)
        {
            clumpFrame = RwFrameCreate();
            RwFrameAddChild(clumpFrame, frame);
            RwMatrixSetIdentity(RwFrameGetMatrix(clumpFrame));
            RpClumpSetFrame(clump, clumpFrame);
            RwFrameUpdateObjects(clumpFrame);
            globalData->hierarchyRootFrame = clumpFrame;
        }
        else
        {
            RpClumpSetFrame(clump, frame);
            RwFrameUpdateObjects(frame);
            globalData->hierarchyRootFrame = frame;
        }

        /*  
            Process our sceneNode array and generate a RenderWare frame hierarchy and/or skeleton
            from it.
        */
        nodeCounter     = 0;
        nodeIterator    = nodes->begin();
        nodeAnimFlags   = (RwUInt32 *)RwMalloc(sizeof(RwUInt32) * nodes->size());
        nodeTags        = (RwInt32 *)RwMalloc(sizeof(RwUInt32) * nodes->size());
        nodeInvMatrices = (RwMatrix *)RwMalloc(sizeof(RwMatrix) * nodes->size());

        processHierarchy(frame, clump);

        if (globalData->m_exportRpHAnim)
        {
            RpHAnimHierarchy    *hierarchy;
            RwUInt32            flags = 0;

            if (globalData->m_rpHAnimNoMatrices)
            {
                flags |= rpHANIMHIERARCHYNOMATRICES;
            }

            if (globalData->m_rpHAnimLocalMatrices)
            {
                flags |= rpHANIMHIERARCHYLOCALSPACEMATRICES;
            }

            hierarchy = RpHAnimHierarchyCreate(nodes->size(), nodeAnimFlags, nodeTags,
                            (RpHAnimHierarchyFlag)flags, rpHANIMSTDKEYFRAMESIZE);

            RpHAnimSetHierarchy(frame, hierarchy);
        }

        /*
            Finally, traverse the scene creating RenderWare atomics and attaching
            them to the frames.
        */
        processGeometry(clump);

        /* Output some info on the clump */
        atomicCount = 0;
        for (nodeIterator = nodes->begin(), i = 0; nodeIterator != nodes->end(); nodeIterator++, i++)
        {
            if ((*nodeIterator)->atomic != NULL)
            {
                atomicCount++;
            }

            if ((*nodeIterator)->mdtIndex == globalData->selectedMdtShape)
            {
                selectionRootIndex = i;
            }
        }

        printf("Hierarchy has %d frames, %d un-skinned atomic(s) and %d skinned atomic(s)\n", nodes->size(), atomicCount, skinNodes->size());
    
        /* If necessary warn about the PS2 64 bone limit */
        if (globalData->m_exportRpSkinWeighting && (nodes->size() > 64))
        {
            printf("Warning: As this clump has more than 64 RpSkin bones it may not work on Playstation 2\n");
        }

        if (globalData->m_rpHAnimSubHierarchy)
        {
            printf("RpHAnim sub-hierarchy starts at frame index %d\n", nodeTags[selectionRootIndex]);
        }
    }
    else
    {
        printf("No objects were selected, export failed\n");
        return NULL;
    }

    /* Process the animation on any tagged hierarchies in the scene */
    if (globalData->m_animSeq)
    {
        int numShapes = DtShapeGetCount();

        for (int shape = 0; shape < numShapes; shape++)
        {
            if (DtShapeGetParentID(shape) == -1 &&
                shape != rootMdtShape)
            {
                char *shapeName, *animName;
                DtShapeGetName(shape, &shapeName);
                animName = strdup(shapeName);
                addAnimToClump(shape, clump, animName);
                free(animName);
            }
        }
    }

    if (globalData->m_exportRpAnim)
    {
        _rpAnimClumpForAllFramesAddSequences(clump);
    }

    return clump;
}

void exportScene::preProcessHierarchy(int mdtNodeIndex)
{
    int         i;
    int         numChildren;
    int         *childArray;
    sceneNode   *newSceneNode;
    bool        foundTag;
    int         nodeTag;

    if (mdtNodeIndex == globalData->selectedMdtShape)
    {
        globalData->underSelectedMdtShape = true;
    }

    /* Figure out how many children this node has */
    DtShapeGetChildren(mdtNodeIndex, &numChildren, &childArray);
    
    /* Create a new node */
    newSceneNode = new(sceneNode)(mdtNodeIndex, numChildren, false);

    /* Add the node to our list */
    nodes->push_back(newSceneNode);

    /*
        Check if this node has any tags on it. If so then we'll need to
        start any auto-tags after it.
    */
    foundTag = getIntTagFromShape(mdtNodeIndex, "RwObjectTag", &nodeTag);

    if ((foundTag == true) && (nodeTag >= tagOffset))
    {
        tagOffset = nodeTag + 1;
    }

    nodeCounter++;

    /* Process children of the node */
    for (i = 0; i < numChildren; i++)
    {
        preProcessHierarchy(childArray[i]);        
    }
    
    if (mdtNodeIndex == globalData->selectedMdtShape)
    {
        globalData->underSelectedMdtShape = false;
    }
}

void exportScene::optimiseHierarchy()
{
    nodeIt      nodeIt0, nodeIt1;
    sceneNode   *node0;

    /* Attempt to merge the nodes in our hierarchy */
    mergeNodes(nodes);

    /* Check if we can remove any redundant nodes in the hierarchy */
    if (globalData->m_optimiseHierarchy)
    {
        /*
            Make sure all scene nodes have keyframes at the same times. We need
            common animation data when passing animation down the hierarchy
        */
        unifyKeyframes();

        for (nodeIt0 = nodes->begin(); nodeIt0 != nodes->end();)
        {
            if (optimiseSceneNode(nodeIt0) == true)
            {
                node0 = *nodeIt0;
                printf("Optimising out node named : \"%s\"\n", node0->name);
                nodeIt1 = nodeIt0++;
                removeSceneNode(nodeIt1);
            }
            else
            {
                nodeIt0++;
            }
        }
    }
}

void exportScene::mergeNodes(sceneNodes *nodeList)
{
    nodeIt      nodeIt0, nodeIt1;
    sceneNode   *node0, *node1;
    
    /*
        Run through the nodes in the scene and record which ones can be
        merged togther.
    */
    for (nodeIt0 = nodeList->begin(); nodeIt0 != nodeList->end(); nodeIt0++)
    {
        node0 = *nodeIt0;
        
        /* All nodes have themselves in their merge list */
        node0->mergedNodes.push_back(node0);

        if (node0->mergeGroup != -1)
        {
            for (nodeIt1 = nodeIt0, nodeIt1++; nodeIt1 != nodeList->end(); nodeIt1++)
            {
                node1 = *nodeIt1;

                if (canMergeNodes(node0, node1))
                {
                    if (node1->numRemappedVertsAfterMerge > 0)
                    {
                        node0->numRemappedVertsAfterMerge += node1->numRemappedVerts;
                        node1->numRemappedVertsAfterMerge = 0;
                        node0->numRemappedFacesAfterMerge += node1->numRemappedFaces;
                        node1->numRemappedFacesAfterMerge = 0;

                        node0->mergedNodes.push_back(node1);
                    }
                }
            }
        }
    }  
}

bool exportScene::canMergeNodes(sceneNode *node0, sceneNode *node1)
{
    if (node0->mergeGroup != node1->mergeGroup)
    {
        return false;
    }

    if (node0->isSkinned != node1->isSkinned)
    {
        return false;
    }

    if (node0->isLit != node1->isLit)
    {
        return false;
    }

    if (node0->isTextured != node1->isTextured)
    {
        return false;
    }

    if (node0->exportRpHAnim != node1->exportRpHAnim)
    {
        return false;
    }

    return true;
}

void exportScene::unifyKeyframes()
{
    nodeIt      nodeIt0;
    sceneNode   *node0;
    animKeyList unifiedKeys;

    for (nodeIt0 = nodes->begin(); nodeIt0 != nodes->end(); nodeIt0++)
    {
        node0 = *nodeIt0;

        unifiedKeys.copyKeyFrames(node0->animKeys, unifiedKeys,
            node0->animKeys.keys.begin()->time, false);
    }

    for (nodeIt0 = nodes->begin(); nodeIt0 != nodes->end(); nodeIt0++)
    {
        node0 = *nodeIt0;

        node0->animKeys.copyKeyFrames(unifiedKeys, node0->animKeys,
            unifiedKeys.keys.begin()->time, false);
    }
}


bool exportScene::optimiseSceneNode(nodeIt nodeIt0)
{
    sceneNode   *node0, *node1;
    nodeIt      nodeIt1;
    int         nodeTag;

    node0 = *nodeIt0;

    /* Only delete unknown nodes */
    if (node0->type != unknownNode)
    {
        return false;
    }
    
	/* Don't delete the node if it has "RwObjectTag" tag */
    if (getIntTagFromShape(node0->mdtIndex, "RwObjectTag", &nodeTag))
    {
        return false;
    }

	/*
        Don't delete the root node if it has more than one child (we
        only process a single hierarchy).
    */
    if ((nodeIt0 == nodes->begin()) && (node0->numChildren > 1))
    {
        return false;
    }

    /*
        Don't delete the node if it has rotation animation and has children which
        are offset. This is because the pivot rotation animation would
        end up interpolating as translation.
    */
    if (node0->animKeys.hasRotationAnimation() == true)
    {
        for (nodeIt1 = nodeIt0; nodeIt1 != nodes->end(); nodeIt1++)
        {
            node1 = *nodeIt1;

            if (node1->parentMdtIndex == node0->mdtIndex)
            {
                RwMatrix *matrix = RwFrameGetMatrix(node1->frame);
                animKeyIt key;

                if (matrix->pos.x != 0.0f || matrix->pos.y != 0.0f ||
                    matrix->pos.z != 0.0f)
                {
                    return false;
                }

                for (key = node1->animKeys.keys.begin(); key != node1->animKeys.keys.end(); key++)
                {
                    if (key->matrix.pos.x != 0.0f || key->matrix.pos.y != 0.0f ||
                        key->matrix.pos.z != 0.0f)
                    {
                        return false;
                    }
                }
            }
        }
    }

    return true;
}

void exportScene::removeSceneNode(nodeIt nodeIt0)
{
    sceneNode   *node0 = *nodeIt0;
    nodeIt      nodeIt1;
    sceneNode   *node1;
    
    /*
        First find any children of the node and flow through the node
        animation and update their parent indices
    */
    if (node0->numChildren > 0)
    {
        for (nodeIt1 = nodeIt0; nodeIt1 != nodes->end(); nodeIt1++)
        {
            node1 = *nodeIt1;

            if (node1->parentMdtIndex == node0->mdtIndex)
            {
                *node1 *= *node0;
                node1->parentMdtIndex = node0->parentMdtIndex;
            }
        }
    }

    /* Now find the parent of the node and update it's child count */
    for (nodeIt1 = nodes->begin(); nodeIt1 != nodes->end(); nodeIt1++)
    {
        node1 = *nodeIt1;

        if (node1->mdtIndex == node0->parentMdtIndex)
        {
            node1->numChildren -= 1;
            node1->numChildren += node0->numChildren;
        }
    }

    delete (*nodeIt0);
    
    nodes->erase(nodeIt0);
}

void exportScene::processHierarchy(RwFrame *parentFrame, RpClump *clump)
{
    int             i;
    RpAtomic        *atomic = NULL;
    RwMatrix        *matrix;
    sceneNode       *node = *nodeIterator;
    unsigned int    localNodeIndex;
    int             nodeTag;
    bool            foundTag;

    /* If we've been passed a parent frame then make this node a child */
    if (parentFrame != NULL)
    {
        RwFrameAddChild(parentFrame, node->frame);
        RwFrameUpdateObjects(node->frame);
    }

    /*
        Reset the current frame to the animation start - it may have been
        changed when processing animation on previous frames in the hierarchy.
    */
    MAnimControl::setCurrentTime(MTime(globalData->m_animStart, MTime::uiUnit()));

    /*
        If this is the root shape and we're exporting in world space
        then put it's modelling matrix on the hierarchy root frame
    */
    if ((nodeCounter == 0) && (globalData->m_coordinateSystem == WORLD))
    {
        matrix = RwFrameGetMatrix(globalData->hierarchyRootFrame);
        node->computeRwMatrix(matrix, false);
    }

    /* Get the tag on this object. If it doesn't exist then set a default */
    foundTag = getIntTagFromShape(node->mdtIndex, "RwObjectTag", &nodeTag);

    if ((foundTag == false) || (nodeTag == -1))
    {
        nodeTag = nodeCounter + tagOffset;
    }

    /* Update some of our per-node data. */    
    nodeTags[nodeCounter]       = nodeTag;
    nodeAnimFlags[nodeCounter]  = rpSKINANIMATEVERTEXGROUP | rpSKINPUSHPARENTMATRIX;

    /* RpHAnim processing */
    if (globalData->m_exportRpHAnim)
    {
        RpHAnimFrameSetID(node->frame, nodeTag);
        
        if (node->exportRpHAnim)
        {
            node->animKeys.addToHierarchyKeyFrames(rpHAnimHierarchyKeys, numAnimatedNodes);
        }
    }
    
    /* RpSkin animation processing */
    if (globalData->m_exportRpSkin)
    {
        node->animKeys.addToHierarchyKeyFrames(rpSkinHierarchyKeys, numAnimatedNodes);
    }

    if (globalData->m_exportRpSkin || node->exportRpHAnim)
    {
        numAnimatedNodes++;
    }

    /* RpSkin geometry linking processing */
    if (globalData->m_exportRpSkinWeighting)
    {
        RwMatrix    *invHierarchyRootMatrix;
        RwMatrix    *rootRelativeMatrix;
        RwMatrix    *LTM;
        
        /* Calculate and store the inverse node matrix within the hierarchy. */
        invHierarchyRootMatrix = RwMatrixCreate();
        rootRelativeMatrix = RwMatrixCreate();
        LTM = RwFrameGetLTM(node->frame);
        
        RwMatrixInvert(invHierarchyRootMatrix, RwFrameGetLTM(globalData->hierarchyRootFrame));
        RwMatrixMultiply(rootRelativeMatrix, LTM, invHierarchyRootMatrix);
        RwMatrixInvert(&nodeInvMatrices[nodeCounter], rootRelativeMatrix);

        RwMatrixDestroy(invHierarchyRootMatrix);
        RwMatrixDestroy(rootRelativeMatrix);

        /*
            Check if this bone dag path is in our list of influence objects
            if so store the bone index so we can link them up later.
        */
        _dagPathBoneIndexList *tempDag = globalData->gDagPathBoneIndexList;
        while (tempDag)
        {
            if (*tempDag->object == node->dagPath)
            {
                tempDag->index = nodeCounter;
            }
            tempDag = tempDag->next;
        }
    }
    
    /* RpAnim processing */
    if (globalData->m_exportRpAnim)
    {
        _rpAnimSeqSetFrameTag(node->frame, nodeTag);

        node->animKeys.convertToRpAnim(node->frame, globalData->m_defaultAnimName);
    }

    /* Increment our counters */
    localNodeIndex = nodeCounter;
    nodeCounter++;
    nodeIterator++;

    /* Process all children recursively */
    if (node->numChildren > 0)
    {        
        for (i = 0; i < node->numChildren; i++)
        {
            processHierarchy(node->frame, clump);        
        }
    }

    if (node->numChildren == 0)
    {
        /* Last node processed had no children so or in the POP flag */
        assert(rpHANIMPOPPARENTMATRIX == rpSKINPOPPARENTMATRIX);
        nodeAnimFlags[localNodeIndex] |= rpHANIMPOPPARENTMATRIX;
    }
    else
    {
        /* Don't do the PUSH on my last child */
        assert(rpHANIMPUSHPARENTMATRIX == rpSKINPUSHPARENTMATRIX);
        nodeAnimFlags[lastChildIndex] &= ~rpHANIMPUSHPARENTMATRIX;
    }

    /* Last child processed was me */
    lastChildIndex = localNodeIndex;
}

void exportScene::processGeometry(RpClump *clump)
{
    nodeIt      nodeIt0;
    sceneNode   *node0;

    for (nodeIt0 = nodes->begin(); nodeIt0 != nodes->end(); nodeIt0++)
    {
        node0 = *nodeIt0;

        if ((node0->numRemappedVertsAfterMerge != 0) &&
            (node0->numRemappedFacesAfterMerge != 0))
        {
            if (!node0->isSkinned)
            {
                node0->generateRwAtomic();
                if (node0->atomic)
                {
                    RpAtomicSetFrame(node0->atomic, node0->frame);

                    RpClumpAddAtomic(clump, node0->atomic);
                }
            }
        }
    }

    /*
        Now the skeleton is fully processed run through the scene converting shapes
        which are linked to the skeleton through skinning.
    */
    if (globalData->m_exportRpSkinWeighting)
    {
        processSkinnedShapes(clump);
    }
}

void exportScene::processSkinnedShapes(RpClump *clump)
{
    int         numShapes = DtShapeGetCount();
    int         shapeIndex;
    sceneNode   *newSceneNode;
    nodeIt      nodeIt0;

    /* Parse the scene and construct a list of skinned nodes. */
    for (shapeIndex = 0; shapeIndex < numShapes; shapeIndex++)
    {
        MStatus             status;
        MDagPath            shapeDAG;
        MObject             shapeObj;
        _skinClusterList    *temp = globalData->gSkinClusterList;
        bool                processed = false;

        DtExt_ShapeGetDagPath(shapeIndex, shapeDAG);
        DtExt_ShapeGetShapeNode(shapeIndex, shapeObj);
        shapeDAG.getAPathTo (shapeObj, shapeDAG);

        while(temp && !processed)
        {
            if (shapeDAG == *temp->skin)
            {
                /* This object is skinned so process it */
                newSceneNode = new(sceneNode)(shapeIndex, 0, true);
                RwFrameUpdateObjects(newSceneNode->frame);
                skinNodes->push_back(newSceneNode);
            }
            temp = temp->next;
        }
    }

    /* Attempt to merge the nodes. */
    mergeNodes(skinNodes);

    /*  
        Finally, build skinned atomics for the nodes. The atomics are attached 
        to the hierarchy root frame.
    */
    for (nodeIt0 = skinNodes->begin(); nodeIt0 != skinNodes->end(); nodeIt0++)
    {
        if (((*nodeIt0)->numRemappedVertsAfterMerge > 0) &&
            ((*nodeIt0)->numRemappedFacesAfterMerge > 0))
        {
            (*nodeIt0)->generateSkinnedRwAtomic(clump, nodes->size(),
                        nodeInvMatrices, nodeAnimFlags, nodeTags);
        }
    }
}

int exportScene::findSelectedShape()
{
    int numShapes = DtShapeGetCount();
    int shapeRoot = -1;

    for (int shape = 0; shape < numShapes && shapeRoot == -1; shape++)
    {
        if (isShapeSelected(shape))
        {
            return(shape);
        }
    }

    return -1;
}

int exportScene::findRootShape()
{
    int numShapes = DtShapeGetCount();
    int shapeRoot = -1;

    for (int shape = 0; shape < numShapes && shapeRoot == -1; shape++)
    {
        if (isShapeSelected(shape))
        {
            shapeRoot = shape;
            while (DtShapeGetParentID(shapeRoot) != -1)
            {
                shapeRoot = DtShapeGetParentID(shapeRoot);
            }
        }
    }

    return shapeRoot;
}
bool exportScene::isShapeSelected(int mdtNodeIndex)
{
    /* get global selection list */
    MSelectionList activeList;
    MGlobal::getActiveSelectionList(activeList);

    MDagPath path, sDagPath;
    bool selected = false;

    MItSelectionList iter(activeList, MFn::kDagNode);
    DtExt_ShapeGetDagPath(mdtNodeIndex, path);

    while (!iter.isDone())
    {
        if (iter.getDagPath(sDagPath))
        {
            MString s1, s2;

            s1 = MString(sDagPath.fullPathName());
            s2 = MString(path.fullPathName());
            if (strncmp(s1.asChar(), s2.asChar(), strlen(s1.asChar())) == 0)
            {
                selected = true;
            }
        }
        iter.next();
    }

    return (selected == true);
}

void exportScene::initialiseMayaSdk()
{
    printf("Initialising Maya SDK\n");

    /* Initialize the Dt database */
    DtExt_SceneInit("exportdff");
    
    DtExt_setOutputCameras(false);
    DtExt_setVertexAnimation(globalData->m_morphTargets);
    DtExt_setMultiTexture(false);
    DtExt_setParents(true);

    /* Disabling inline textures causes a crash on Maya 2.0 */
#ifndef MAYA_20
    DtExt_setInlineTextures(false);
#endif

    /* lets have some joint shapes */
    DtExt_setJointHierarchy(true);

    /* Let's have DT collapse all the transforms */
    DtExt_setOutputTransforms(kTRANSFORMALL);

#ifdef VERY_VERBOSE
    if (m_verbose)
    { 
        DtExt_setDebug(DEBUG_GEOMAT);
    }
#endif
    
    if (globalData->m_verbose)
    {
        printf("Generating database\n");
    }

    /*  Set the Maya frame to the animation start. This must be done before
        DtExt_dbInit as this function caches geometry and this never gets
        updated on calls to DtFrameSet (unless vertex animation is on). We
        can't use DtFrameSet itself here as it checks that the database has
        already been cached and starts spitting error messages.
    */
    MAnimControl::setCurrentTime(MTime(globalData->m_animStart, MTime::uiUnit()));

    /* Walk the dag and fill in the internal database */
    DtExt_dbInit();

    DtSetUpdateShaders(false);
    DtFrameSetStart(globalData->m_animStart);
    DtFrameSetEnd(globalData->m_animEnd);
    DtFrameSet(globalData->m_animStart);

    printf("Finished initialising Maya SDK\n");
}

void exportScene::cleanupMayaSdk()
{
    /* Clean up the allocated memory and internal storage */
    DtExt_CleanUp();
}


void exportScene::intialiseGlobalSceneData()
{
    if (globalData->m_exportRpSkinWeighting)
    {
        /* search for skin clusters and assign each 'bone' a unique ID */
        printf("Checking scene for skinClusters\n");

        forAllNodesInScene(MFn::kSkinClusterFilter, skinClusterCallBack, NULL);
    }

    printf("Checking scene for IK setups\n");

    forAllNodesInScene(MFn::kIkHandle, ikHandleCallBack, NULL);

#if 0
    /* Find all blend shapes in the scene and store how they are made up */
    forAllNodesInScene(MFn::kBlendShape, blendShapeCallBack, NULL);

    /* Find all nurbs surfaces in the scene and store how they are made up */
    forAllNodesInScene(MFn::kNurbsSurface, nurbsSurfaceCallBack, NULL);
#endif
    
    /* calculate a new scale factor if scale to is selected */
    if (globalData->m_scaleType == SCALETO)
    {
        globalData->m_scaleFactor = calculateScaleToFactor();
    }
    if (globalData->m_verbose)
    {
        printf("Scale to factor is %f\n", globalData->m_scaleFactor);
    }

    /*
        Build a material map. This processes all materials in the scene and detects
        which ones are identical. Materials can then be shared across objects.
    */
    globalData->materialMap = BuildMaterialMap(NULL, globalData->m_verbose, &globalData->m_textureNameWarning);
}

void exportScene::cleanupGlobalSceneData()
{
    blindDataTemplateIt bdIt;

    /* free the global skin cluster list */
    _skinClusterList *SCtemp = globalData->gSkinClusterList;
    while(SCtemp)
    {
        globalData->gSkinClusterList = SCtemp->next;
        
        delete SCtemp->skin;
        delete SCtemp->skinCluster;
        delete SCtemp->joints;
        free(SCtemp);

        SCtemp = globalData->gSkinClusterList;
    }

    /* free the global skin cluster influence tag list */
    _dagPathBoneIndexList *DBItemp = globalData->gDagPathBoneIndexList;
    while(DBItemp)
    {
        globalData->gDagPathBoneIndexList = DBItemp->next;
        free(DBItemp);
        DBItemp = globalData->gDagPathBoneIndexList;
    }

    /* free the global ik handle list */
    _ikHandleList *IHLtemp = globalData->gIkHandleList;
    while(IHLtemp)
    {
        globalData->gIkHandleList = IHLtemp->next;
        free(IHLtemp);
        IHLtemp = globalData->gIkHandleList;
    }

    /* free the material map */
    if (globalData->materialMap != NULL)
    {
        DestroyMaterialMap(globalData->materialMap);
        globalData->materialMap = NULL;
    }

    /* free the blind data template names */
    for (bdIt = globalData->blindDataTemplates.begin(); bdIt != globalData->blindDataTemplates.end(); bdIt++)
    {
        free (bdIt->templateNodeName);
        free (bdIt->templateName);
    }
}

float exportScene::calculateScaleToFactor()
{
    int             numShapes, shape;
    MDagPath        dagPath;
    MFnDagNode      *dagNode;
    double          newScale;
    MBoundingBox    *clumpBounds = NULL;

    if (globalData->m_verbose)
    {
        printf("Calculating a scale to factor\n");
    }

    DtExt_SceneInit("exportdff scaleto calculation");
    DtExt_setOutputCameras(false);

    /* Let's have DT collapse all the transforms */
    DtExt_setOutputTransforms(kTRANSFORMNONE);

    /* Walk the dag and fill in the internal database */
    DtExt_dbInit();

    numShapes = DtShapeGetCount();

    for (shape = 0; shape < numShapes; shape++)
    {

        DtExt_ShapeGetDagPath(shape, dagPath);
        dagNode = new MFnDagNode(dagPath);

        if (clumpBounds == NULL)
        {
            clumpBounds = new MBoundingBox(dagNode->boundingBox().min(),
                                           dagNode->boundingBox().max());
        }
        else
        {
            clumpBounds->expand(dagNode->boundingBox().min());
            clumpBounds->expand(dagNode->boundingBox().max());
        }
    }

    newScale = clumpBounds->width();

    if (clumpBounds->height() > newScale)
    {
        newScale = clumpBounds->height();
    }

    if (clumpBounds->depth() > newScale)
    {
        newScale = clumpBounds->depth();
    }

    /* Clean up the allocated memory and internal storage */
    DtExt_CleanUp();    

    return (globalData->m_scaleFactor / (float)newScale);
}

MObject exportScene::skinClusterCallBack(MObject object, void *pData)
{
    /* For each skinCluster node, get the list of influence objects */
    MFnSkinCluster  skinCluster(object);
    MDagPathArray   infs;
    MStatus         stat;
    unsigned int    nInfs = skinCluster.influenceObjects(infs, &stat);
    unsigned int    nGeoms = skinCluster.numOutputConnections();
    unsigned int    i;

    if (nInfs == 0)
    {
        stat = MS::kFailure;
    }

    /* uniquely tag the influence objects */
    for (i = 0; i < nInfs; i++)
    {
        _dagPathBoneIndexList *temp = globalData->gDagPathBoneIndexList;

        while (temp)
        {
            if (infs[i] == *temp->object)
            {
                break;
            }

            temp = temp->next;
        }

        if (!temp)
        {
            /* add a unique tag for this influence object */
            _dagPathBoneIndexList *newEntry = (_dagPathBoneIndexList *)malloc(sizeof(_dagPathBoneIndexList));
            
            newEntry->next      = globalData->gDagPathBoneIndexList;
            newEntry->index     = -1;
            newEntry->object    = new MDagPath(infs[i]);
            globalData->gDagPathBoneIndexList   = newEntry;
        }
    }

    /*
        Loop through the geometries affected by this cluster
        adding an entry to our list for each.
    */
    for (i = 0; i < nGeoms; i++)
    {
        unsigned int index = skinCluster.indexForOutputConnection(i, &stat);
        unsigned int i;

        /* get the dag path of the i'th geometry */
        MDagPath skinPath;
        stat = skinCluster.getPathAtIndex(index,skinPath);

        /* create a new skincluster list entry */
        _skinClusterList *newEntry = (_skinClusterList *)malloc(sizeof(_skinClusterList));

        newEntry->next                      = globalData->gSkinClusterList;
        globalData->gSkinClusterList        = newEntry;
        
        newEntry->skin                      = new MDagPath(skinPath);
        newEntry->skinCluster               = new MFnSkinCluster(object);
        newEntry->joints                    = new MDagPathArray;

        /* Copy the array of influence objects */
        for (i = 0; i < nInfs; i++)
        {
            newEntry->joints->append(infs[i]);
        }
    }

    return object;
}

MObject exportScene::ikHandleCallBack(MObject object, void *pData)
{
    MFnIkHandle ikHandle(object);

    MDagPath startJoint;
    MDagPath endEffector;

    ikHandle.getStartJoint(startJoint);
    ikHandle.getEffector(endEffector);

    _ikHandleList *newEntry = (_ikHandleList *)malloc(sizeof(_ikHandleList));
    
    newEntry->next          = globalData->gIkHandleList;
    newEntry->startJoint    = new MDagPath(startJoint);
    newEntry->endEffector   = new MDagPath(endEffector);
    newEntry->handle        = new MObject(object);
    
    globalData->gIkHandleList = newEntry;

    return object;
}


#if 0
MObject exportScene::blendShapeCallBack(MObject object, void *pData)
{
    int                 i, j, k, numBaseObjects, numTargetObjects;
    unsigned int        numWeights;
    MStatus             status;
    MObjectArray        baseObjects;
    MObjectArray        targetObjects;

    MFnBlendShapeDeformer deformer(object, &status);

    status = deformer.getBaseObjects(baseObjects);

    if (status != MStatus::kSuccess)
    {
        return object;
    }

    numBaseObjects = baseObjects.length();

    for (i = 0; i < numBaseObjects; i++)
    {
        int numTargetsOnObject = 0;
        MString baseObjectName = ObjectName(baseObjects[i]);

        printf("\n\nBase Object = %s\n", baseObjectName.asChar());

        numWeights = deformer.numWeights(&status);

        for (j = 0; j < numWeights; j++)
        {
            deformer.getTargets(baseObjects[i], j, targetObjects);
            
            numTargetObjects = targetObjects.length();
            
            if (numTargetObjects > 0)
            {
                for (k = 0; k < numTargetObjects; k++)
                {
                    MString targetObjectName = ObjectName(targetObjects[k]);
                    printf("Target Object = %s\n", targetObjectName.asChar());
                }

                numTargetsOnObject++;
            }
        }
    }
    
    return object;
}

MObject exportScene::nurbsSurfaceCallBack(MObject object, void *pData)
{
    int         cvsU, cvsV, knotsU, knotsV;
    bool        isBezier;
    MStatus     status;

    MFnNurbsSurface surface(object, &status);

    isBezier = surface.isBezier(&status);

    if (status != MStatus::kSuccess)
    {
        return object;
    }

    printf("isBezier = %s\n", isBezier ? "TRUE" : "FALSE");
    
    cvsU    = surface.numCVsInU(&status);
    cvsV    = surface.numCVsInV(&status);
    knotsU  = surface.numKnotsInU(&status);
    knotsV  = surface.numKnotsInV(&status);
    
    printf("Surface has %d * %d CVs and %d * %d knots\n", cvsU, cvsV, knotsU, knotsV);
    
    return object;
}
#endif

bool exportScene::addAnimToClump(int mdtNodeIndex, RpClump *clump, RwChar *animName)
{
    /* recurse down shape hierarchy pulling out tags, find
       frame in clump with tag and call add anim to frame */
    TagData tagData;
    RwFrame *frame;
    int     tag = -1;
    int     numChildren, child, *childArray;
    bool    foundTag;

    if (mdtNodeIndex == -1)
    {
        return false;
    }

    frame = RpClumpGetFrame(clump);

    foundTag = getIntTagFromShape(mdtNodeIndex, "RwObjectTag", &tag);

    if (foundTag && (tag != -1) && frame)
    {
        tagData.tag = tag;
        tagData.frame = NULL;

        RwFrameForAllChildren(frame, _findTaggedFrame,
                              (void *)&tagData);

        if (tagData.frame)
        {
            sceneNode node(mdtNodeIndex, 0, false);
            node.animKeys.convertToRpAnim(tagData.frame, animName);
        }
    }

    /* convert the selected clumps */
    DtShapeGetChildren(mdtNodeIndex, &numChildren, &childArray);
    for (child = 0; child < numChildren; child++)
    {
        int childShape = childArray[child];

        addAnimToClump(childShape, clump, animName);
    }

    return true;
}

/* General Functions */
MString ObjectName(MObject object)
{
    MStatus             status;
    MString             name;
    MFnDependencyNode   depNode(object, &status);
    
    if (status != MStatus::kSuccess)
    {
        return name;
    }
    
    name = depNode.name(&status);
    
    return name;
}

RwFrame * _findTaggedFrame(RwFrame *frame, void *data)
{
    TagData *tagData;

    tagData = (TagData *)data;

    if (_rpAnimSeqGetFrameTag(frame) == tagData->tag)
    {
        tagData->frame = frame;
        return (NULL);
    }

    RwFrameForAllChildren(frame,
                          _findTaggedFrame,
                          data);

    return (frame);
}

RpAnimSequence *_rpAnimClumpAddSequenceCallback(RpAnimSequence *animSequence, void *data)
{
    RpAnimClumpAddSequence((RpClump *)data,
        RpAnimSequenceGetName(animSequence),
        NULL,
        NULL);

    return (animSequence);
}

RwFrame *_rpAnimClumpAddAllSequencesCallBack(RwFrame *frame, void *data)
{
    RpAnimFrameForAllSequences(frame,
        _rpAnimClumpAddSequenceCallback,
        data);

    RwFrameForAllChildren(frame,
        _rpAnimClumpAddAllSequencesCallBack,
        data);

    return (frame);
}

RpClump *_rpAnimClumpForAllFramesAddSequences(RpClump *clump)
{
    if (clump)
    {
        RwFrame *clumpFrame = RpClumpGetFrame(clump);

        if (clumpFrame)
        {
            _rpAnimClumpAddAllSequencesCallBack(clumpFrame, (void *) clump);
        }

        return (clump);
    }

    return (NULL);
}